package easik.ui.tree;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import easik.Easik;
import easik.sketch.attribute.EntityAttribute;
import easik.sketch.attribute.UniqueKey;
import easik.sketch.constraint.CommutativeDiagram;
import easik.sketch.constraint.Constraint;
import easik.sketch.constraint.ProductConstraint;
import easik.sketch.constraint.PullbackConstraint;
import easik.sketch.constraint.SumConstraint;
import easik.sketch.path.SketchPath;
import easik.sketch.vertex.EntityNode;
import easik.ui.menu.AddCommutativeDiagramAction;
import easik.ui.menu.AddProductConstraintAction;
import easik.ui.menu.AddPullbackConstraintAction;
import easik.ui.menu.AddSumConstraintAction;
import easik.ui.menu.popup.NewEntityAction;
import easik.ui.tree.popup.AddAttributeAction;
import easik.ui.tree.popup.AddPathAction;
import easik.ui.tree.popup.AddUniqueKeyAction;
import easik.ui.tree.popup.DeleteAttributeAction;
import easik.ui.tree.popup.DeleteConstraintAction;
import easik.ui.tree.popup.DeleteEntityAction;
import easik.ui.tree.popup.DeleteUniqueKeyAction;
import easik.ui.tree.popup.EditAttributeAction;
import easik.ui.tree.popup.EditUniqueKeyAction;
import easik.ui.tree.popup.HideConstraint;
import easik.ui.tree.popup.DeletePathAction;
import easik.ui.tree.popup.RenameEntityAction;
import easik.ui.tree.popup.ShowConstraint;


/**
 * Class to display the information tree.  This tree contains information about entities, attributes 
 * and constraints.
 * 
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @version 2006-08-02 Kevin Green
 */
public class InfoTreeUI extends JScrollPane {
	
	/**
	 * The information tree
	 */
	private JTree _infoTree;
	/**
	 * The information tree model
	 */
	private DefaultTreeModel _infoTreeModel;
	/**
	 * The top node of the tree
	 */
	private DefaultMutableTreeNode _topNode;
	/**
	 * The entities tree node
	 */
	private NonLeafNode _tree_entities;
	/**
	 * The constraints tree node
	 */
	private NonLeafNode _tree_constraints;
	/**
	 * The commutative diagram tree node
	 */
	private NonLeafNode _tree_constraints_commutative;
	/**
	 * The product constraint tree node
	 */
	private NonLeafNode _tree_constraints_product;
	/**
	 * The pullback constraint tree node
	 */
	private NonLeafNode _tree_constraints_pullback;
	/**
	 * The sum constraint tree node
	 */
	private NonLeafNode _tree_constraints_sum;
	/**
	 * The popup menu for the tree
	 */
	private JPopupMenu _popupMenu;
	/**
	 * The tree popup menu position
	 */
	private Point _popupPosition;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _addAttributeItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _editAttributeItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _deleteAttributeItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _addUniqueKeyItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _editUniqueKeyItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _deleteUniqueKeyItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _showConstraintItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _hideConstraintItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _deleteConstraintItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _addEntityItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _renameEntityItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _deleteEntityItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _addCommutativeItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _addProductItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _addPullbackItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _addSumItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _addPathItem;
	/**
	 * Tree popup menu item
	 */
	private JMenuItem _deletePathItem;
	
	/**
	 * Default constructor
	 */
	public InfoTreeUI(){
		//Create the top node.
	    _topNode = new DefaultMutableTreeNode("EA Sketch");
	    
	    //Create a tree that allows one selection at a time.
	    _infoTreeModel = new DefaultTreeModel(_topNode);
	    _infoTree = new JTree(_infoTreeModel);
	    _infoTree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
	    _infoTree.setRootVisible(false); //Hides root node
	    _infoTree.setShowsRootHandles(true); //Shows nodes off root
	    
	    DefaultTreeCellRenderer renderer = new DefaultTreeCellRenderer();
	    renderer.setOpenIcon(null);
	    renderer.setClosedIcon(null);
	    renderer.setLeafIcon(null);
	    _infoTree.setCellRenderer(renderer);
	    
	    
		//Add tree into scroll pane and then to this pane
		this.setViewportView(_infoTree);
		this.setMinimumSize(new Dimension(150, 200));
		
		_popupMenu = new JPopupMenu();
		_popupPosition = new Point(0, 0);
		buildPopupMenu();
	    
	    //Create Category Heads
	    _tree_entities = new NonLeafNode("Entities");
	    _topNode.add(_tree_entities);
	    
	    _tree_constraints = new NonLeafNode("Constraints");
	    _topNode.add(_tree_constraints);

	    _tree_constraints_commutative = new NonLeafNode("Commutative Diagrams");
	    _tree_constraints.add(_tree_constraints_commutative);

	    _tree_constraints_product = new NonLeafNode("Product Constraints");
	    _tree_constraints.add(_tree_constraints_product);

	    _tree_constraints_pullback = new NonLeafNode("Pullback Constraints");
	    _tree_constraints.add(_tree_constraints_pullback);
	    
	    _tree_constraints_sum = new NonLeafNode("Sum Constraints");
	    _tree_constraints.add(_tree_constraints_sum);
}
	
	/**
	 * Creates the popup menu
	 */
	private void buildPopupMenu() {
		_popupMenu.add(_addAttributeItem = new JMenuItem(new AddAttributeAction()));
		_popupMenu.add(_editAttributeItem = new JMenuItem(new EditAttributeAction()));
		_popupMenu.add(_deleteAttributeItem = new JMenuItem(new DeleteAttributeAction()));
		_popupMenu.add(_addUniqueKeyItem = new JMenuItem(new AddUniqueKeyAction()));
		_popupMenu.add(_editUniqueKeyItem = new JMenuItem(new EditUniqueKeyAction()));
		_popupMenu.add(_deleteUniqueKeyItem = new JMenuItem(new DeleteUniqueKeyAction()));
		_popupMenu.add(_showConstraintItem = new JMenuItem(new ShowConstraint()));
		_popupMenu.add(_hideConstraintItem = new JMenuItem(new HideConstraint()));
		_popupMenu.add(_deleteConstraintItem = new JMenuItem(new DeleteConstraintAction()));
		_popupMenu.add(_addEntityItem = new JMenuItem(new NewEntityAction(new Point(50,50))));
		_popupMenu.add(_renameEntityItem = new JMenuItem(new RenameEntityAction()));
		_popupMenu.add(_deleteEntityItem = new JMenuItem(new DeleteEntityAction()));
		_popupMenu.add(_addCommutativeItem = new JMenuItem(new AddCommutativeDiagramAction()));
		_popupMenu.add(_addProductItem = new JMenuItem(new AddProductConstraintAction()));
		_popupMenu.add(_addPullbackItem = new JMenuItem(new AddPullbackConstraintAction()));
		_popupMenu.add(_addSumItem = new JMenuItem(new AddSumConstraintAction()));
		_popupMenu.add(_addPathItem = new JMenuItem(new AddPathAction()));
		_popupMenu.add(_deletePathItem = new JMenuItem(new DeletePathAction()));
		
		_infoTree.addMouseListener(new MouseAdapter() {

			public void mousePressed(MouseEvent evt) {
				if (evt.isPopupTrigger()) {
					_infoTree.setSelectionRow(_infoTree.getClosestRowForLocation(evt.getX(), evt.getY()));
					_popupPosition.setLocation(evt.getX(), evt.getY());
					if(setPopMenuItems()){
						_popupMenu.show(evt.getComponent(), evt.getX(), evt.getY());
					}
				}
			}
			public void mouseReleased(MouseEvent evt) {
				if (evt.isPopupTrigger()) {
					_infoTree.setSelectionRow(_infoTree.getClosestRowForLocation(evt.getX(), evt.getY()));
					_popupPosition.setLocation(evt.getX(), evt.getY());
					if(setPopMenuItems()){
						_popupMenu.show(evt.getComponent(), evt.getX(), evt.getY());
					}
				}
			}
		});
	}
	
	/**
	 * Sets which of the menu items will be visible
	 * 
	 * @return true if the popup should be displayed, false otherwise
	 */
	public boolean setPopMenuItems(){
		//If there is nothing seleceted then just do nothing
		if(Easik.getInstance().getFrame().getInfoTreeUI().getInfoTree().isSelectionEmpty())
		{
			return false;
		}
		
		//Get currently selected object
		DefaultMutableTreeNode curSelectedNode = (DefaultMutableTreeNode) 
					Easik.getInstance().getFrame().getInfoTreeUI().getInfoTree().getSelectionPath().getLastPathComponent();
		Object curSelected = curSelectedNode.getUserObject();
		
		//Hide all elements
		for(Component c : _popupMenu.getComponents()){
			c.setVisible(false);
		}
		
		//Check what is currently selected
		if(curSelected instanceof EntityNode){
			_addAttributeItem.setVisible(true);
			_addUniqueKeyItem.setVisible(true);
			_renameEntityItem.setVisible(true);
			_deleteEntityItem.setVisible(true);
		}
		else if(curSelected instanceof EntityAttribute){
			_editAttributeItem.setVisible(true);
			_deleteAttributeItem.setVisible(true);
		}
		else if(curSelected instanceof UniqueKey){
			_editUniqueKeyItem.setVisible(true);
			_deleteUniqueKeyItem.setVisible(true);
		}
		else if(curSelected instanceof Constraint){
			if(curSelected instanceof SumConstraint || curSelected instanceof ProductConstraint){
				_addPathItem.setVisible(true);
			}
			if(((Constraint)curSelected).isVisible()){
				_hideConstraintItem.setVisible(true);
			}
			else{
				_showConstraintItem.setVisible(true);
			}
			_deleteConstraintItem.setVisible(true);
		}
		else if(curSelected instanceof SketchPath){
			Object myConst = ((DefaultMutableTreeNode)curSelectedNode.getParent()).getUserObject();
			if(myConst instanceof SumConstraint || myConst instanceof ProductConstraint){
				_deletePathItem.setVisible(true);
			}
		}
		else if(curSelectedNode == _tree_entities){
			_addEntityItem.setVisible(true);
		}
		else if(curSelectedNode == _tree_constraints){
			_addCommutativeItem.setVisible(true);
			_addProductItem.setVisible(true);
			_addPullbackItem.setVisible(true);
			_addSumItem.setVisible(true);
		}
		else if(curSelectedNode == _tree_constraints_commutative){
			_addCommutativeItem.setVisible(true);
		}
		else if(curSelectedNode == _tree_constraints_product){
			_addProductItem.setVisible(true);
		}
		else if(curSelectedNode == _tree_constraints_pullback){
			_addPullbackItem.setVisible(true);
		}
		else if(curSelectedNode == _tree_constraints_sum){
			_addSumItem.setVisible(true);
		}
		else{
			return false;
		}
		return true;
	}
	
	/**
	 * Refreshes the tree so updates visualize properly
	 */
	public void refreshTree(){
		_infoTreeModel.reload(_topNode);
	}
	
	/**
	 * Refreshes the tree so updates visualize properly
	 * 
	 * @param inNode The node to be refreshed
	 */
	public void refreshTree(DefaultMutableTreeNode inNode){
		_infoTreeModel.reload(inNode);
	}
	
	/**
	 * Returns the information tree model
	 * 
	 * @return The information tree model
	 */
	public DefaultTreeModel getInfoTreeModel(){
		return _infoTreeModel;
	}
	
	/**
	 * Returns the information tree
	 * 
	 * @return The information tree
	 */
	public JTree getInfoTree(){
		return _infoTree;
	}
	
	/**
	 * Returns the Entities tree node
	 * 
	 * @return The Entities tree node
	 */
	public DefaultMutableTreeNode getEntities(){
		return _tree_entities;
	}
	
	/**
	 * Returns the Contraints tree node
	 * 
	 * @return The Contraints tree node
	 */
	public DefaultMutableTreeNode getContraints(){
		return _tree_constraints;
	}
	
	/**
	 * Adds an entity and its attributes to the tree
	 * 
	 * @param inEntity The entity to be added to the tree
	 */
	public void addEntity(EntityNode inEntity){
		//Create entity Node
		NonLeafNode myEntityNode = new NonLeafNode(inEntity);
		_tree_entities.add(myEntityNode);
		inEntity.setNode(myEntityNode);
		NonLeafNode myKeyNode = new NonLeafNode("Key(s)");
		myEntityNode.add(myKeyNode);
		inEntity.setKeyNode(myKeyNode);
		
		//Add entity attributes to tree
		int numAttributes = inEntity.getAttributes().size();
		for(int i=0; i<numAttributes; i++){
			DefaultMutableTreeNode attNode = new DefaultMutableTreeNode(inEntity.getAttributes().get(i));
			((EntityAttribute)inEntity.getAttributes().get(i)).setNode(attNode);
			myEntityNode.add(attNode);
		}
		
		//Add unique keys to tree
		int numUniqueKeys = inEntity.getUniqueKeys().size();
		for(int i=0; i<numUniqueKeys; i++){
			DefaultMutableTreeNode keyNode = new DefaultMutableTreeNode(inEntity.getUniqueKeys().get(i));
			((UniqueKey)inEntity.getUniqueKeys().get(i)).setNode(keyNode);
			myKeyNode.add(keyNode);
		}
		inEntity.testKeysNode();
		refreshTree(_tree_entities);
		_infoTree.scrollPathToVisible(new TreePath(myEntityNode.getPath()));
	}
	
	/**
	 * Removes an entity from the tree
	 * 
	 * @param toRemove Entity to be removed from the tree
	 */
	public void removeEntity(EntityNode toRemove){
		_tree_entities.remove(toRemove.getNode());
		refreshTree(_tree_entities);
	}
	
	/**
	 * Add a constraint to the info tree.
	 * 
	 * @param c The constraint to add
	 * @since 2006-05-30 Vera Ranieri
	 */
	public void addConstraint(Constraint c){
		NonLeafNode constraint = new NonLeafNode(c);
		if(c instanceof ProductConstraint)
			_tree_constraints_product.add(constraint);
		else if( c instanceof SumConstraint)
			_tree_constraints_sum.add(constraint);
		else if(c instanceof PullbackConstraint)
			_tree_constraints_pullback.add(constraint);
		else if(c instanceof CommutativeDiagram)
			_tree_constraints_commutative.add(constraint);
	
		//Add the paths to the constraint
		int numPaths = c.getPaths().size();
		for(int i=0; i<numPaths; i++){
			DefaultMutableTreeNode myNode = new DefaultMutableTreeNode((SketchPath)c.getPaths().get(i));
			constraint.add(myNode);
		}
		
		c.setNode(constraint);
		
		refreshTree((DefaultMutableTreeNode)constraint.getParent());
		_infoTree.scrollPathToVisible(new TreePath(constraint.getPath()));
	}
	
	/**
	 * Remove a constraint from the info tree.
	 * 
	 * @param c The constraint to remove
	 * @since 2006-05-30 Vera Ranieri
	 */
	public void removeConstraint(Constraint c){
		_tree_constraints.remove(c.getNode());
		_infoTreeModel.reload();
	}
}
